<?php
/**
 * Tags
 *
 * @package     AutomatorWP\Calculator\Tags
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

// GLOBAL TAGS

/**
 * Calculator tags
 *
 * @since 1.0.0
 *
 * @param array $tags The tags
 *
 * @return array
 */
function automatorwp_calculator_get_tags( $tags ) {

    $tags['calculator'] = array(
        'label' => 'Calculator',
        'tags'  => array(),
        'icon'  => AUTOMATORWP_CALCULATOR_URL . 'assets/calculator.svg',
    );

    $tags['calculator']['tags']['calculate( FORMULA )'] = array(
        'label'     => __( 'Calculate', 'automatorwp-pro' ),
        'type'      => 'text',
        'preview'   => __( 'Calculates the given formula, replace "FORMULA" by the formula of your choice.', 'automatorwp-pro' ),
    );

    return $tags;

}
add_filter( 'automatorwp_get_tags', 'automatorwp_calculator_get_tags' );

/**
 * Calculator tags names
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_calculator_get_tags_names() {

    $tags = automatorwp_calculator_get_tags( array() );
    $tags_names = array();

    foreach( $tags['calculator']['tags'] as $tag => $args ) {
        $tags_names[] = explode( '(', $tag )[0];
    }

    return $tags_names;

}

/**
 * Skip tags replacements
 *
 * @since 1.0.0
 *
 * @param bool      $skip           If tag should get skipped, by default false
 * @param string    $tag_name       The tag name (without "{}")
 * @param int       $automation_id  The automation ID
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 *
 * @return bool
 */
function automatorwp_calculator_skip_tag_replacement( $skip, $tag_name, $automation_id, $user_id, $content ) {

    foreach( automatorwp_calculator_get_tags_names() as $tag ) {
        if( automatorwp_starts_with( $tag_name, $tag . '(' ) ) {
            return true;
        }
    }

    return $skip;

}
add_filter( 'automatorwp_skip_tag_replacement', 'automatorwp_calculator_skip_tag_replacement', 10, 5 );

/**
 * Parse tags
 *
 * @since 1.0.0
 *
 * @param string    $parsed_content     Content parsed
 * @param array     $replacements       Automation replacements
 * @param int       $automation_id      The automation ID
 * @param int       $user_id            The user ID
 * @param string    $content            The content to parse
 *
 * @return string
 */
function automatorwp_calculator_post_parse_automation_tags( $parsed_content, $replacements, $automation_id, $user_id, $content ) {

    $replacements = array();

    if( empty( $parsed_content ) ) {
        return $parsed_content;
    }

    // Get the functions
    preg_match_all( "/\{\s*(.*?)\s*\(\s*(.*)\s*\)\s*\}/", $parsed_content, $matches );

    if( is_array( $matches ) && isset( $matches[1] ) ) {

        foreach( $matches[1] as $i => $function ) {
            // Skip if not is our function
            if( ! in_array( $function, automatorwp_calculator_get_tags_names() ) ) {
                continue;
            }

            $value = automatorwp_parse_automation_tags( $automation_id, $user_id, $matches[2][$i] );

            switch ( $function ) {
                case 'calculate':
                    $value = automatorwp_calculator_calculate_result( $value );
                    break;
            }

            $replacements[$matches[0][$i]] = $value;
        }

    }

    if( count( $replacements ) > 0 ) {
        $tags = array_keys( $replacements );

        // Replace all tags by their replacements
        $parsed_content = str_replace( $tags, $replacements, $parsed_content );
    }

    return $parsed_content;
}
add_filter( 'automatorwp_post_parse_automation_tags', 'automatorwp_calculator_post_parse_automation_tags', 10, 5 );

// ACTION TAGS

/**
 * Formula result tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_calculator_get_actions_formula_result_tags() {

    return array(
        'formula_result' => array(
            'label'     => __( 'Formula result', 'automatorwp-pro' ),
            'type'      => 'integer',
            'preview'   => '123',
        ),
    );

}

/**
 * Custom action operation result tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $action         The action object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last action log object
 *
 * @return string
 */
function automatorwp_calculator_get_action_formula_result_tag_replacement( $replacement, $tag_name, $action, $user_id, $content, $log ) {


    $action_args = automatorwp_get_action( $action->type );

    // Skip if action is not from this integration
    if( $action_args['integration'] !== 'calculator' ) {
        return $replacement;
    }

    switch( $tag_name ) {
        case 'formula_result':
            $replacement = automatorwp_get_log_meta( $log->id, 'formula_result', true );
            break;
    }

    return $replacement;

}
add_filter( 'automatorwp_get_action_tag_replacement', 'automatorwp_calculator_get_action_formula_result_tag_replacement', 10, 6 );
